import Foundation

/// Risk factor coefficients based on academic research
/// References: Jensen 2013, Kodama 2009, Doll 2004
struct RiskFactor {
    let name: String
    let hazardRatio: Double
    let coefficient: Double
    let source: String
    let confidenceInterval: (lower: Double, upper: Double)?
    
    init(name: String, hazardRatio: Double, source: String = "", confidenceInterval: (lower: Double, upper: Double)? = nil) {
        // Input validation and bounds checking
        guard hazardRatio > 0 else {
            fatalError("Hazard ratio must be positive")
        }
        guard hazardRatio >= 0.1 && hazardRatio <= 10.0 else {
            fatalError("Hazard ratio out of reasonable bounds (0.1-10.0)")
        }
        
        self.name = name
        self.hazardRatio = hazardRatio
        self.coefficient = log(hazardRatio)
        self.source = source
        self.confidenceInterval = confidenceInterval
    }
    
    /// Calculate years impact on life expectancy from hazard ratio
    /// Uses more sophisticated conversion based on actuarial principles
    func calculateYearsImpact(baseLifeExpectancy: Double, currentAge: Double = 40.0) -> Double {
        guard baseLifeExpectancy > 0 && currentAge >= 0 && currentAge <= 120 else {
            return 0.0 // Invalid inputs return no impact
        }
        
        // More sophisticated conversion using survival function integration
        // Based on proportional hazards model principles
        let remainingYears = max(baseLifeExpectancy - currentAge, 0)
        
        if hazardRatio == 1.0 {
            return 0.0 // No impact
        }
        
        // Calculate life expectancy change using integral approximation
        // LE_new ≈ LE_base * (1 - (HR - 1) * adjustment_factor)
        let adjustmentFactor = min(0.25, max(0.05, remainingYears / 100.0))
        let proportionalChange = (hazardRatio - 1.0) * adjustmentFactor
        
        return -proportionalChange * remainingYears
    }
    
    /// Calculate impact multiplier for mortality rate adjustments
    /// Used by integration engine to apply risk factor effects
    func calculateImpactMultiplier(healthMetrics: HealthMetrics) -> Double {
        // For this basic implementation, return the hazard ratio directly
        // More sophisticated implementations could consider health metrics
        // to determine if the risk factor applies to this individual
        
        // Example logic for different risk factors
        switch name.lowercased() {
        case "smoking":
            // Smoking impact varies by current status and history
            return hazardRatio
        case "obesity", "bmi":
            // BMI-based risk factor
            if let bmi = healthMetrics.bmi, bmi > 30.0 {
                return hazardRatio
            } else if let bmi = healthMetrics.bmi, bmi > 25.0 {
                return 1.0 + (hazardRatio - 1.0) * 0.5 // Reduced impact for overweight
            }
            return 1.0 // No impact if normal weight
        case "exercise", "cardio", "vo2max":
            // Exercise/cardio fitness impact
            if let vo2Max = healthMetrics.vo2Max, vo2Max < 35.0 {
                return hazardRatio
            } else if let vo2Max = healthMetrics.vo2Max, vo2Max > 50.0 {
                return 1.0 / hazardRatio // Protective effect for high fitness
            }
            return 1.0 // Neutral for average fitness
        case "heart_rate", "resting_heart_rate":
            // Resting heart rate impact
            if let rhr = healthMetrics.restingHeartRate, Double(rhr) > 80.0 {
                return hazardRatio
            }
            return 1.0
        default:
            // Default: apply hazard ratio directly
            return hazardRatio
        }
    }
    
    // MARK: - Research-Based Risk Factors
    
    // Smoking Risk Factors (Doll 2004)
    static let currentSmoker = RiskFactor(
        name: "Current Smoking",
        hazardRatio: 2.8,
        source: "Doll et al. 2004",
        confidenceInterval: (lower: 2.5, upper: 3.2)
    )
    
    static let heavySmoker = RiskFactor(
        name: "Heavy Smoking (>20 cigs/day)",
        hazardRatio: 3.2,
        source: "Doll et al. 2004",
        confidenceInterval: (lower: 2.8, upper: 3.7)
    )
    
    static let formerSmoker = RiskFactor(
        name: "Former Smoker (>5 years quit)",
        hazardRatio: 1.3,
        source: "Doll et al. 2004",
        confidenceInterval: (lower: 1.1, upper: 1.5)
    )
    
    // Obesity Risk Factors (Jensen 2013)
    static let mildObesity = RiskFactor(
        name: "Mild Obesity (BMI 30-35)",
        hazardRatio: 1.2,
        source: "Jensen et al. 2013",
        confidenceInterval: (lower: 1.1, upper: 1.4)
    )
    
    static let severeObesity = RiskFactor(
        name: "Severe Obesity (BMI >35)",
        hazardRatio: 1.8,
        source: "Jensen et al. 2013",
        confidenceInterval: (lower: 1.5, upper: 2.2)
    )
    
    static let underweight = RiskFactor(
        name: "Underweight (BMI <18.5)",
        hazardRatio: 1.4,
        source: "Jensen et al. 2013",
        confidenceInterval: (lower: 1.2, upper: 1.7)
    )
    
    // Exercise and Fitness (Kodama 2009)
    static let highFitness = RiskFactor(
        name: "High Cardiorespiratory Fitness",
        hazardRatio: 0.7,
        source: "Kodama et al. 2009",
        confidenceInterval: (lower: 0.6, upper: 0.8)
    )
    
    static let moderateExercise = RiskFactor(
        name: "Regular Moderate Exercise",
        hazardRatio: 0.8,
        source: "Kodama et al. 2009",
        confidenceInterval: (lower: 0.7, upper: 0.9)
    )
    
    static let sedentaryLifestyle = RiskFactor(
        name: "Sedentary Lifestyle",
        hazardRatio: 1.3,
        source: "Multiple studies",
        confidenceInterval: (lower: 1.2, upper: 1.5)
    )
    
    // Cardiovascular Health
    static let lowRestingHR = RiskFactor(
        name: "Low Resting Heart Rate (<60 bpm)",
        hazardRatio: 0.85,
        source: "Cardiovascular studies",
        confidenceInterval: (lower: 0.75, upper: 0.95)
    )
    
    static let highRestingHR = RiskFactor(
        name: "High Resting Heart Rate (>90 bpm)",
        hazardRatio: 1.4,
        source: "Cardiovascular studies",
        confidenceInterval: (lower: 1.2, upper: 1.7)
    )
    
    static let excellentHRV = RiskFactor(
        name: "Excellent Heart Rate Variability",
        hazardRatio: 0.75,
        source: "HRV mortality studies",
        confidenceInterval: (lower: 0.65, upper: 0.85)
    )
    
    // Sleep and Lifestyle
    static let optimalSleep = RiskFactor(
        name: "Optimal Sleep (7-8 hours)",
        hazardRatio: 0.9,
        source: "Sleep mortality studies",
        confidenceInterval: (lower: 0.85, upper: 0.95)
    )
    
    static let chronicSleepDeprivation = RiskFactor(
        name: "Chronic Sleep Deprivation (<5 hours)",
        hazardRatio: 1.6,
        source: "Sleep mortality studies",
        confidenceInterval: (lower: 1.3, upper: 2.0)
    )
    
    static let highPhysicalActivity = RiskFactor(
        name: "High Physical Activity (>10k steps/day)",
        hazardRatio: 0.8,
        source: "Activity studies",
        confidenceInterval: (lower: 0.7, upper: 0.9)
    )
}

/// Risk factor assessment based on health metrics
struct RiskAssessment {
    let appliedFactors: [RiskFactor]
    let totalHazardRatio: Double
    let yearsImpact: Double
    let riskScore: RiskScore
    
    enum RiskScore {
        case veryLow, low, moderate, high, veryHigh
        
        static func from(hazardRatio: Double) -> RiskScore {
            switch hazardRatio {
            case 0.0..<0.8: return .veryLow
            case 0.8..<0.9: return .low
            case 0.9..<1.1: return .moderate
            case 1.1..<1.5: return .high
            default: return .veryHigh
            }
        }
    }
    
    static func assess(metrics: HealthMetrics, baseLifeExpectancy: Double, currentAge: Double = 40.0) -> RiskAssessment {
        var factors: [RiskFactor] = []
        var totalCoefficient = 0.0
        
        // Enhanced BMI assessment with multiple categories
        if let bmi = metrics.bmi {
            switch bmi {
            case 0..<18.5:
                factors.append(RiskFactor.underweight)
                totalCoefficient += RiskFactor.underweight.coefficient
            case 30..<35:
                factors.append(RiskFactor.mildObesity)
                totalCoefficient += RiskFactor.mildObesity.coefficient
            case 35...100:
                factors.append(RiskFactor.severeObesity)
                totalCoefficient += RiskFactor.severeObesity.coefficient
            default:
                break // Normal BMI range (18.5-30)
            }
        }
        
        // Enhanced resting heart rate assessment
        if let rhr = metrics.restingHeartRate {
            switch rhr {
            case 0..<60:
                factors.append(RiskFactor.lowRestingHR)
                totalCoefficient += RiskFactor.lowRestingHR.coefficient
            case 90...200:
                factors.append(RiskFactor.highRestingHR)
                totalCoefficient += RiskFactor.highRestingHR.coefficient
            default:
                break // Normal range (60-90)
            }
        }
        
        // VO2 Max and fitness assessment
        if let vo2 = metrics.vo2Max {
            switch vo2 {
            case 50...100: // Excellent fitness
                factors.append(RiskFactor.highFitness)
                totalCoefficient += RiskFactor.highFitness.coefficient
            case 35..<50: // Moderate fitness
                factors.append(RiskFactor.moderateExercise)
                totalCoefficient += RiskFactor.moderateExercise.coefficient
            case 0..<25: // Poor fitness
                factors.append(RiskFactor.sedentaryLifestyle)
                totalCoefficient += RiskFactor.sedentaryLifestyle.coefficient
            default:
                break // Average fitness range
            }
        }
        
        // Enhanced sleep assessment
        if let sleep = metrics.sleepHours {
            switch sleep {
            case 7...8:
                factors.append(RiskFactor.optimalSleep)
                totalCoefficient += RiskFactor.optimalSleep.coefficient
            case 0..<5:
                factors.append(RiskFactor.chronicSleepDeprivation)
                totalCoefficient += RiskFactor.chronicSleepDeprivation.coefficient
            default:
                break // Sub-optimal but not extreme
            }
        }
        
        // Enhanced activity assessment
        if let steps = metrics.stepCount {
            if steps >= 12000 { // High activity threshold
                factors.append(RiskFactor.highPhysicalActivity)
                totalCoefficient += RiskFactor.highPhysicalActivity.coefficient
            } else if steps < 3000 { // Sedentary threshold
                factors.append(RiskFactor.sedentaryLifestyle)
                totalCoefficient += RiskFactor.sedentaryLifestyle.coefficient
            }
        }
        
        // HRV assessment (if available)
        if let hrv = metrics.hrv, hrv > 50 { // Assuming HRV in milliseconds
            factors.append(RiskFactor.excellentHRV)
            totalCoefficient += RiskFactor.excellentHRV.coefficient
        }
        
        // Calculate total hazard ratio using exponential of sum of coefficients
        let totalHR = exp(totalCoefficient)
        
        // Calculate total years impact with proper age adjustment
        let yearsImpact = factors.reduce(0.0) { sum, factor in
            sum + factor.calculateYearsImpact(baseLifeExpectancy: baseLifeExpectancy, currentAge: currentAge)
        }
        
        let riskScore = RiskScore.from(hazardRatio: totalHR)
        
        return RiskAssessment(
            appliedFactors: factors,
            totalHazardRatio: totalHR,
            yearsImpact: yearsImpact,
            riskScore: riskScore
        )
    }
}

// MARK: - Risk Factor Validation Extension
extension RiskFactor {
    /// Validate that hazard ratio is within scientifically reasonable bounds
    var isValid: Bool {
        return hazardRatio > 0.1 && hazardRatio < 10.0
    }
    
    /// Get risk category based on hazard ratio
    var riskCategory: String {
        switch hazardRatio {
        case 0.0..<0.8: return "Protective"
        case 0.8..<1.0: return "Slightly Protective"
        case 1.0: return "Neutral"
        case 1.0..<1.2: return "Slightly Increased Risk"
        case 1.2..<2.0: return "Moderate Risk"
        default: return "High Risk"
        }
    }
}